<?php

use App\Http\Controllers\Api\AuthController;
use App\Http\Controllers\Api\StaffController;
use App\Http\Controllers\Api\VisitorController;
use App\Http\Controllers\Api\VisitorNoteController;
use App\Http\Controllers\Api\SettingsController;
use App\Http\Controllers\Api\AuditLogController;
use App\Http\Controllers\Api\RealtimeController;
use App\Http\Controllers\Api\ScheduledVisitController;
use App\Http\Controllers\Api\AnalyticsController;
use App\Http\Controllers\Api\BackupController;
use App\Http\Controllers\Api\BadgeController;
use Illuminate\Support\Facades\Route;

Route::prefix('v1')->group(function () {
    // Public routes (no authentication required) - with rate limiting
    Route::middleware(['throttle:10,1'])->group(function () {
        Route::post('/auth/register', [AuthController::class, 'register']);
        Route::post('/auth/login', [AuthController::class, 'login']);
        Route::post('/auth/forgot-password', [AuthController::class, 'forgotPassword']);
        Route::post('/auth/reset-password', [AuthController::class, 'resetPassword']);
    });
    
    // Public visitor check-in (for tablet kiosk) - moderate rate limit
    Route::middleware(['throttle:30,1'])->group(function () {
        Route::post('/visitors', [VisitorController::class, 'store']);
        Route::get('/staff', [StaffController::class, 'index']);
        // Public scheduled visit booking
        Route::post('/scheduled-visits', [ScheduledVisitController::class, 'store']);
    });

    // Protected routes (require authentication) - higher rate limit
    Route::middleware(['auth:api', 'throttle:60,1'])->group(function () {
        // Auth routes
        Route::post('/auth/logout', [AuthController::class, 'logout']);
        Route::post('/auth/refresh', [AuthController::class, 'refresh']);
        Route::get('/auth/me', [AuthController::class, 'me']);

        // Visitor routes (admin only)
        // Specific routes must come before parameterized routes
        Route::get('/visitors/hourly-stats', [VisitorController::class, 'hourlyStats']);
        Route::get('/visitors/checked-in/list', [VisitorController::class, 'checkedIn']);
        Route::get('/visitors/checked-out/list', [VisitorController::class, 'checkedOut']);
        Route::get('/visitors/search', [VisitorController::class, 'search']);
        Route::get('/visitors/export', [VisitorController::class, 'export']);
        Route::post('/visitors/bulk-checkout', [VisitorController::class, 'bulkCheckout']);
        Route::post('/visitors/bulk-delete', [VisitorController::class, 'bulkDelete']);
        Route::get('/visitors', [VisitorController::class, 'index']);
        Route::get('/visitors/{id}', [VisitorController::class, 'show']);
        Route::post('/visitors/{id}/checkout', [VisitorController::class, 'checkout']);

        // Visitor Notes routes
        Route::get('/visitors/{visitorId}/notes', [VisitorNoteController::class, 'index']);
        Route::post('/visitors/{visitorId}/notes', [VisitorNoteController::class, 'store']);
        Route::put('/visitors/{visitorId}/notes/{noteId}', [VisitorNoteController::class, 'update']);
        Route::delete('/visitors/{visitorId}/notes/{noteId}', [VisitorNoteController::class, 'destroy']);

        // Staff routes (admin only)
        Route::post('/staff', [StaffController::class, 'store']);
        Route::put('/staff/{id}', [StaffController::class, 'update']);
        Route::delete('/staff/{id}', [StaffController::class, 'destroy']);

        // Settings routes (admin only)
        Route::get('/settings', [SettingsController::class, 'index']);
        Route::post('/settings/change-password', [SettingsController::class, 'changePassword']);
        Route::post('/settings/mail-config', [SettingsController::class, 'saveMailConfig']);
        Route::post('/settings/checkout-message', [SettingsController::class, 'saveCheckoutMessage']);

        // Audit Log routes (admin only)
        Route::get('/audit-logs', [AuditLogController::class, 'index']);
        Route::get('/audit-logs/{id}', [AuditLogController::class, 'show']);
        Route::get('/audit-logs/export', [AuditLogController::class, 'export']);

        // Real-time updates (SSE)
        Route::get('/realtime/stream', [RealtimeController::class, 'stream']);

        // Scheduled Visits routes (admin)
        Route::get('/scheduled-visits', [ScheduledVisitController::class, 'index']);
        Route::get('/scheduled-visits/{id}', [ScheduledVisitController::class, 'show']);
        Route::put('/scheduled-visits/{id}', [ScheduledVisitController::class, 'update']);
        Route::post('/scheduled-visits/{id}/cancel', [ScheduledVisitController::class, 'cancel']);
        Route::post('/scheduled-visits/{id}/check-in', [ScheduledVisitController::class, 'checkIn']);

        // Advanced Analytics routes
        Route::get('/analytics/dashboard', [AnalyticsController::class, 'dashboard']);
        Route::get('/analytics/weekly-trends', [AnalyticsController::class, 'weeklyTrends']);
        Route::get('/analytics/monthly-trends', [AnalyticsController::class, 'monthlyTrends']);

        // Backup routes (admin only)
        Route::post('/backups', [BackupController::class, 'create']);
        Route::get('/backups', [BackupController::class, 'list']);
        Route::get('/backups/{filename}/download', [BackupController::class, 'download']);
        Route::delete('/backups/{filename}', [BackupController::class, 'delete']);

        // Badge routes
        Route::get('/visitors/{visitorId}/badge', [BadgeController::class, 'generate']);
        Route::get('/visitors/{visitorId}/badge/print', [BadgeController::class, 'print']);
        Route::get('/badges/template', [BadgeController::class, 'template']);
    });
});
