<?php

namespace App\Services;

use App\Models\Visitor;
use App\Models\Staff;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;

class NotificationService
{
    /**
     * Send notification to host when visitor arrives
     */
    public static function notifyHostOnArrival(Visitor $visitor): bool
    {
        try {
            $host = $visitor->host;
            if (!$host) {
                return false;
            }
            
            // Try to get email from staff record, or skip if no email
            $hostEmail = $host->email ?? null;
            if (!$hostEmail) {
                return false; // No email for host
            }

            // Load mail config
            self::loadMailConfig();

            // Get notification template
            $message = self::getHostArrivalMessage($visitor);

            Mail::raw($message, function ($mail) use ($visitor, $host) {
                $mail->to($host->email)
                    ->subject("Visitor Arrival: {$visitor->full_name}");
            });

            Log::info("Host notification sent for visitor {$visitor->id} to {$host->email}");
            return true;
        } catch (\Exception $e) {
            Log::error("Failed to send host notification: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Send notification to admin on visitor check-in
     */
    public static function notifyAdminOnCheckIn(Visitor $visitor): bool
    {
        try {
            // Get admin email from settings or config
            $adminEmail = config('mail.from.address');
            if (!$adminEmail) {
                return false;
            }

            self::loadMailConfig();

            $message = self::getAdminCheckInMessage($visitor);

            Mail::raw($message, function ($mail) use ($visitor, $adminEmail) {
                $mail->to($adminEmail)
                    ->subject("New Visitor Check-in: {$visitor->full_name}");
            });

            Log::info("Admin notification sent for visitor {$visitor->id}");
            return true;
        } catch (\Exception $e) {
            Log::error("Failed to send admin notification: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Send notification to visitor on checkout (already implemented, but enhanced)
     */
    public static function notifyVisitorOnCheckout(Visitor $visitor): bool
    {
        if (!$visitor->email) {
            return false;
        }

        try {
            self::loadMailConfig();

            $message = Storage::disk('local')->exists('checkout_message.txt')
                ? Storage::disk('local')->get('checkout_message.txt')
                : 'Dear {visitorName}, thank you for visiting {companyName} today! We hope you had a pleasant experience. Safe travels!';

            // Replace variables
            $message = str_replace('{visitorName}', $visitor->full_name, $message);
            $message = str_replace('{companyName}', $visitor->company_name, $message);
            $message = str_replace('{timeIn}', $visitor->time_in->format('F j, Y g:i A'), $message);
            $message = str_replace('{timeOut}', $visitor->time_out ? $visitor->time_out->format('F j, Y g:i A') : 'N/A', $message);
            $message = str_replace('{hostName}', $visitor->host->name, $message);
            $message = str_replace('{purpose}', $visitor->purpose, $message);

            Mail::raw($message, function ($mail) use ($visitor) {
                $mail->to($visitor->email)
                    ->subject('Thank You for Visiting - Checkout Confirmation');
            });

            return true;
        } catch (\Exception $e) {
            Log::error("Failed to send visitor checkout notification: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Load mail configuration from storage
     */
    private static function loadMailConfig(): void
    {
        if (Storage::disk('local')->exists('mail_config.json')) {
            try {
                $mailConfig = json_decode(Storage::disk('local')->get('mail_config.json'), true);
                if ($mailConfig) {
                    config([
                        'mail.mailers.smtp.host' => $mailConfig['host'] ?? config('mail.mailers.smtp.host'),
                        'mail.mailers.smtp.port' => $mailConfig['port'] ?? config('mail.mailers.smtp.port'),
                        'mail.mailers.smtp.username' => $mailConfig['username'] ?? config('mail.mailers.smtp.username'),
                        'mail.mailers.smtp.password' => $mailConfig['password'] ?? config('mail.mailers.smtp.password'),
                        'mail.mailers.smtp.encryption' => $mailConfig['encryption'] ?? config('mail.mailers.smtp.encryption'),
                        'mail.from.address' => $mailConfig['fromAddress'] ?? config('mail.from.address'),
                        'mail.from.name' => $mailConfig['fromName'] ?? config('mail.from.name'),
                    ]);
                }
            } catch (\Exception $e) {
                Log::warning('Failed to load mail config from storage: ' . $e->getMessage());
            }
        }
    }

    /**
     * Get host arrival message template
     */
    private static function getHostArrivalMessage(Visitor $visitor): string
    {
        return "Hello {$visitor->host->name},\n\n" .
               "You have a visitor:\n\n" .
               "Name: {$visitor->full_name}\n" .
               "Company: {$visitor->company_name}\n" .
               "Purpose: {$visitor->purpose}\n" .
               "Phone: {$visitor->phone_number}\n" .
               "Email: " . ($visitor->email ?: 'N/A') . "\n" .
               "Check-in Time: " . $visitor->time_in->format('F j, Y g:i A') . "\n\n" .
               "Please proceed to reception to meet your visitor.\n\n" .
               "Thank you!";
    }

    /**
     * Get admin check-in notification message
     */
    private static function getAdminCheckInMessage(Visitor $visitor): string
    {
        return "New Visitor Check-in\n\n" .
               "Visitor Details:\n" .
               "Name: {$visitor->full_name}\n" .
               "Company: {$visitor->company_name}\n" .
               "Purpose: {$visitor->purpose}\n" .
               "Host: {$visitor->host->name} ({$visitor->host->department})\n" .
               "Check-in Time: " . $visitor->time_in->format('F j, Y g:i A') . "\n\n" .
               "Visitor ID: {$visitor->id}";
    }

    /**
     * Send confirmation email to visitor for scheduled visit
     */
    public static function notifyVisitorScheduled(\App\Models\ScheduledVisit $visit): bool
    {
        if (!$visit->visitor_email) {
            return false;
        }

        try {
            self::loadMailConfig();

            $message = "Dear {$visit->visitor_name},\n\n" .
                      "Your visit has been scheduled:\n\n" .
                      "Date: " . $visit->scheduled_date->format('F j, Y') . "\n" .
                      "Time: " . date('g:i A', strtotime($visit->scheduled_time)) . "\n" .
                      "Host: {$visit->host->name} ({$visit->host->department})\n" .
                      "Purpose: {$visit->purpose}\n" .
                      "Company: {$visit->company_name}\n\n" .
                      "Please arrive on time. If you need to cancel or reschedule, please contact us.\n\n" .
                      "We look forward to seeing you!\n\n" .
                      "Confirmation Token: {$visit->confirmation_token}";

            Mail::raw($message, function ($mail) use ($visit) {
                $mail->to($visit->visitor_email)
                    ->subject("Visit Scheduled: {$visit->scheduled_date->format('M j, Y')} at " . date('g:i A', strtotime($visit->scheduled_time)));
            });

            return true;
        } catch (\Exception $e) {
            Log::error("Failed to send scheduled visit confirmation: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Notify host of scheduled visit
     */
    public static function notifyHostScheduled(\App\Models\ScheduledVisit $visit): bool
    {
        $host = $visit->host;
        $hostEmail = $host->email ?? null;
        
        if (!$hostEmail) {
            return false;
        }

        try {
            self::loadMailConfig();

            $message = "Hello {$host->name},\n\n" .
                      "You have a scheduled visit:\n\n" .
                      "Visitor: {$visit->visitor_name}\n" .
                      "Company: {$visit->company_name}\n" .
                      "Purpose: {$visit->purpose}\n" .
                      "Date: " . $visit->scheduled_date->format('F j, Y') . "\n" .
                      "Time: " . date('g:i A', strtotime($visit->scheduled_time)) . "\n" .
                      "Phone: {$visit->visitor_phone}\n" .
                      "Email: " . ($visit->visitor_email ?: 'N/A') . "\n\n" .
                      ($visit->notes ? "Notes: {$visit->notes}\n\n" : "") .
                      "Please be available to meet your visitor at the scheduled time.\n\n" .
                      "Thank you!";

            Mail::raw($message, function ($mail) use ($visit, $hostEmail) {
                $mail->to($hostEmail)
                    ->subject("Scheduled Visit: {$visit->visitor_name} on " . $visit->scheduled_date->format('M j, Y'));
            });

            return true;
        } catch (\Exception $e) {
            Log::error("Failed to send host scheduled visit notification: " . $e->getMessage());
            return false;
        }
    }
}
