<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Http\Controllers\Api\RealtimeController;
use App\Models\Visitor;
use App\Services\AuditLogService;
use App\Services\NotificationService;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Storage;

class VisitorController extends Controller
{
    /**
     * Display a listing of visitors with advanced filters.
     */
    public function index(Request $request): JsonResponse
    {
        $query = Visitor::with('host');

        // Filter by status
        if ($request->has('status') && $request->status !== 'all') {
            $query->where('status', $request->status);
        }

        // Date range filter - Time In
        if ($request->has('date_from')) {
            $query->whereDate('time_in', '>=', $request->date_from);
        }
        if ($request->has('date_to')) {
            $query->whereDate('time_in', '<=', $request->date_to);
        }

        // Filter by host
        if ($request->has('host_id')) {
            $query->where('host_id', $request->host_id);
        }

        // Filter by purpose
        if ($request->has('purpose')) {
            $query->where('purpose', $request->purpose);
        }

        // Advanced search functionality
        if ($request->has('search') && $request->search) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('full_name', 'like', "%{$search}%")
                  ->orWhere('phone_number', 'like', "%{$search}%")
                  ->orWhere('email', 'like', "%{$search}%")
                  ->orWhere('company_name', 'like', "%{$search}%")
                  ->orWhereHas('host', function ($hostQuery) use ($search) {
                      $hostQuery->where('name', 'like', "%{$search}%")
                                ->orWhere('department', 'like', "%{$search}%");
                  });
            });
        }

        // Sorting
        $sortBy = $request->get('sort_by', 'time_in');
        $sortOrder = $request->get('sort_order', 'desc');
        $query->orderBy($sortBy, $sortOrder);

        // Pagination - default to 50 per page, max 1000
        $perPage = min((int) $request->get('per_page', 50), 1000);
        $page = (int) $request->get('page', 1);
        
        // If per_page is very high (like 1000), use simple pagination for better performance
        if ($perPage >= 1000) {
            $visitors = $query->get();
            return response()->json([
                'success' => true,
                'data' => $visitors->map(function ($visitor) {
                    return $this->formatVisitor($visitor);
                }),
                'meta' => [
                    'total' => $visitors->count(),
                    'per_page' => $perPage,
                    'current_page' => 1,
                ],
            ]);
        }

        $paginated = $query->paginate($perPage, ['*'], 'page', $page);

        return response()->json([
            'success' => true,
            'data' => $paginated->map(function ($visitor) {
                return $this->formatVisitor($visitor);
            }),
            'meta' => [
                'total' => $paginated->total(),
                'per_page' => $paginated->perPage(),
                'current_page' => $paginated->currentPage(),
                'last_page' => $paginated->lastPage(),
            ],
        ]);
    }

    /**
     * Store a newly created visitor (check-in).
     */
    public function store(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'fullName' => 'required|string|max:255',
            'phoneNumber' => 'required|string|max:20',
            'email' => 'nullable|email|max:255',
            'companyName' => 'required|string|max:255',
            'purpose' => 'required|string|max:255',
            'hostId' => 'required|exists:staff,id',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors(),
            ], 422);
        }

        $visitor = Visitor::create([
            'full_name' => $request->fullName,
            'phone_number' => $request->phoneNumber,
            'email' => $request->email,
            'company_name' => $request->companyName,
            'purpose' => $request->purpose,
            'host_id' => $request->hostId,
            'time_in' => now(),
            'status' => 'checked-in',
        ]);

        $visitor->load('host');

        // Log action
        AuditLogService::log('created', $visitor, null, $visitor->toArray(), 'Visitor checked in', $request);

        // Send notifications
        try {
            NotificationService::notifyHostOnArrival($visitor);
            NotificationService::notifyAdminOnCheckIn($visitor);
        } catch (\Exception $e) {
            \Log::warning('Failed to send notifications: ' . $e->getMessage());
            // Don't fail the check-in if notifications fail
        }

        // Broadcast real-time update
        try {
            RealtimeController::broadcast('visitor.checked_in', [
                'visitor' => $this->formatVisitor($visitor),
            ]);
        } catch (\Exception $e) {
            \Log::warning('Failed to broadcast real-time update: ' . $e->getMessage());
        }

        return response()->json([
            'success' => true,
            'message' => 'Visitor checked in successfully',
            'data' => $this->formatVisitor($visitor),
        ], 201);
    }

    /**
     * Display the specified visitor.
     */
    public function show(string $id): JsonResponse
    {
        $visitor = Visitor::with('host')->findOrFail($id);

        return response()->json([
            'success' => true,
            'data' => $this->formatVisitor($visitor),
        ]);
    }

    /**
     * Check out a visitor.
     */
    public function checkout(string $id): JsonResponse
    {
        $visitor = Visitor::findOrFail($id);

        if ($visitor->status === 'checked-out') {
            return response()->json([
                'success' => false,
                'message' => 'Visitor is already checked out',
            ], 400);
        }

        $visitor->update([
            'time_out' => now(),
            'status' => 'checked-out',
        ]);

        $visitor->load('host');

        // Log action
        AuditLogService::log('updated', $visitor, ['status' => 'checked-in'], ['status' => 'checked-out'], 'Visitor checked out', $request);

        // Send checkout email if email is provided
        if ($visitor->email) {
            try {
                NotificationService::notifyVisitorOnCheckout($visitor);
            } catch (\Exception $e) {
                // Log error but don't fail the checkout
                \Log::error('Failed to send checkout email: ' . $e->getMessage());
            }
        }

        // Broadcast real-time update
        try {
            RealtimeController::broadcast('visitor.checked_out', [
                'visitor' => $this->formatVisitor($visitor),
            ]);
        } catch (\Exception $e) {
            \Log::warning('Failed to broadcast real-time update: ' . $e->getMessage());
        }

        return response()->json([
            'success' => true,
            'message' => 'Visitor checked out successfully',
            'data' => $this->formatVisitor($visitor),
        ]);
    }

    /**
     * Get checked-in visitors.
     */
    public function checkedIn(): JsonResponse
    {
        $visitors = Visitor::with('host')
            ->where('status', 'checked-in')
            ->orderBy('time_in', 'desc')
            ->get();

        return response()->json([
            'success' => true,
            'data' => $visitors->map(function ($visitor) {
                return $this->formatVisitor($visitor);
            }),
        ]);
    }

    /**
     * Get checked-out visitors.
     */
    public function checkedOut(): JsonResponse
    {
        $visitors = Visitor::with('host')
            ->where('status', 'checked-out')
            ->orderBy('time_in', 'desc')
            ->get();

        return response()->json([
            'success' => true,
            'data' => $visitors->map(function ($visitor) {
                return $this->formatVisitor($visitor);
            }),
        ]);
    }

    /**
     * Search visitors by name or phone (for check-out).
     */
    public function search(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'query' => 'required|string|min:2',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors(),
            ], 422);
        }

        $query = $request->query;
        $visitors = Visitor::with('host')
            ->where('status', 'checked-in')
            ->where(function ($q) use ($query) {
                $q->where('full_name', 'like', "%{$query}%")
                  ->orWhere('phone_number', 'like', "%{$query}%")
                  ->orWhere('email', 'like', "%{$query}%");
            })
            ->orderBy('time_in', 'desc')
            ->get();

        return response()->json([
            'success' => true,
            'data' => $visitors->map(function ($visitor) {
                return $this->formatVisitor($visitor);
            }),
        ]);
    }

    /**
     * Export visitors to CSV.
     */
    public function export(Request $request): \Illuminate\Http\Response
    {
        $query = Visitor::with('host');

        // Apply filters if provided
        if ($request->has('status') && $request->status !== 'all') {
            $query->where('status', $request->status);
        }

        if ($request->has('search') && $request->search) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('full_name', 'like', "%{$search}%")
                  ->orWhere('phone_number', 'like', "%{$search}%")
                  ->orWhere('company_name', 'like', "%{$search}%");
            });
        }

        $visitors = $query->orderBy('time_in', 'desc')->get();

        $filename = 'visitors_' . now()->format('Y-m-d_His') . '.csv';

        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => "attachment; filename=\"{$filename}\"",
        ];

        $callback = function () use ($visitors) {
            $file = fopen('php://output', 'w');
            
            // BOM for Excel compatibility
            fprintf($file, chr(0xEF).chr(0xBB).chr(0xBF));
            
            // Headers
            fputcsv($file, [
                'ID',
                'Full Name',
                'Phone Number',
                'Company Name',
                'Purpose',
                'Host Name',
                'Host Department',
                'Time In',
                'Time Out',
                'Status',
            ]);

            // Data rows
            foreach ($visitors as $visitor) {
                fputcsv($file, [
                    $visitor->id,
                    $visitor->full_name,
                    $visitor->phone_number,
                    $visitor->company_name,
                    $visitor->purpose,
                    $visitor->host->name,
                    $visitor->host->department,
                    $visitor->time_in->format('Y-m-d H:i:s'),
                    $visitor->time_out ? $visitor->time_out->format('Y-m-d H:i:s') : '',
                    $visitor->status,
                ]);
            }

            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }

    /**
     * Format visitor data for API response.
     */
    /**
     * Get hourly visitor statistics for today.
     * Optimized to use database aggregation instead of loading all records.
     */
    public function hourlyStats(): JsonResponse
    {
        $today = now()->startOfDay();
        $tomorrow = now()->copy()->addDay()->startOfDay();

        // Use database aggregation for better performance
        $hourlyCounts = Visitor::whereBetween('time_in', [$today, $tomorrow])
            ->selectRaw('HOUR(time_in) as hour, COUNT(*) as count')
            ->whereRaw('HOUR(time_in) >= 8 AND HOUR(time_in) <= 17')
            ->groupBy('hour')
            ->orderBy('hour')
            ->pluck('count', 'hour')
            ->toArray();

        // Initialize hours array (8 AM to 5 PM)
        $hourlyStats = [];
        for ($hour = 8; $hour <= 17; $hour++) {
            $hourKey = $hour < 12 ? ($hour . 'AM') : ($hour === 12 ? '12PM' : (($hour - 12) . 'PM'));
            $hourlyStats[$hourKey] = $hourlyCounts[$hour] ?? 0;
        }

        // Convert to array format for frontend
        $data = array_map(function ($time, $count) {
            return [
                'time' => $time,
                'visitors' => $count,
            ];
        }, array_keys($hourlyStats), array_values($hourlyStats));

        return response()->json([
            'success' => true,
            'data' => $data,
        ]);
    }

    /**
     * Send checkout email to visitor.
     */
    private function sendCheckoutEmail(Visitor $visitor): void
    {
        // Load mail config from storage if available
        if (Storage::disk('local')->exists('mail_config.json')) {
            try {
                $mailConfig = json_decode(Storage::disk('local')->get('mail_config.json'), true);
                if ($mailConfig) {
                    // Update mail configuration dynamically
                    config([
                        'mail.mailers.smtp.host' => $mailConfig['host'] ?? config('mail.mailers.smtp.host'),
                        'mail.mailers.smtp.port' => $mailConfig['port'] ?? config('mail.mailers.smtp.port'),
                        'mail.mailers.smtp.username' => $mailConfig['username'] ?? config('mail.mailers.smtp.username'),
                        'mail.mailers.smtp.password' => $mailConfig['password'] ?? config('mail.mailers.smtp.password'),
                        'mail.mailers.smtp.encryption' => $mailConfig['encryption'] ?? config('mail.mailers.smtp.encryption'),
                        'mail.from.address' => $mailConfig['fromAddress'] ?? config('mail.from.address'),
                        'mail.from.name' => $mailConfig['fromName'] ?? config('mail.from.name'),
                    ]);
                }
            } catch (\Exception $e) {
                \Log::warning('Failed to load mail config from storage: ' . $e->getMessage());
            }
        }

        // Get checkout message from storage
        $message = Storage::disk('local')->exists('checkout_message.txt')
            ? Storage::disk('local')->get('checkout_message.txt')
            : 'Dear {visitorName}, thank you for visiting {companyName} today! We hope you had a pleasant experience. Safe travels!';

        // Replace variables in message
        $message = str_replace('{visitorName}', $visitor->full_name, $message);
        $message = str_replace('{companyName}', $visitor->company_name, $message);
        $message = str_replace('{timeIn}', $visitor->time_in->format('F j, Y g:i A'), $message);
        $message = str_replace('{timeOut}', $visitor->time_out ? $visitor->time_out->format('F j, Y g:i A') : 'N/A', $message);
        $message = str_replace('{hostName}', $visitor->host->name, $message);
        $message = str_replace('{purpose}', $visitor->purpose, $message);

        // Send email
        Mail::raw($message, function ($mail) use ($visitor) {
            $mail->to($visitor->email)
                ->subject('Thank You for Visiting - Checkout Confirmation');
        });
    }

    /**
     * Bulk checkout visitors.
     */
    public function bulkCheckout(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'visitor_ids' => 'required|array',
            'visitor_ids.*' => 'required|exists:visitors,id',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors(),
            ], 422);
        }

        $visitorIds = $request->visitor_ids;
        $checkedOut = 0;
        $alreadyCheckedOut = 0;
        $errors = [];

        foreach ($visitorIds as $visitorId) {
            try {
                $visitor = Visitor::find($visitorId);
                if (!$visitor) {
                    $errors[] = "Visitor ID {$visitorId} not found";
                    continue;
                }

                if ($visitor->status === 'checked-out') {
                    $alreadyCheckedOut++;
                    continue;
                }

                $visitor->update([
                    'time_out' => now(),
                    'status' => 'checked-out',
                ]);

                // Send email if available
                if ($visitor->email) {
                    try {
                        $this->sendCheckoutEmail($visitor);
                    } catch (\Exception $e) {
                        \Log::error("Failed to send checkout email to {$visitor->email}: " . $e->getMessage());
                    }
                }

                $checkedOut++;
            } catch (\Exception $e) {
                $errors[] = "Failed to checkout visitor ID {$visitorId}: " . $e->getMessage();
            }
        }

        return response()->json([
            'success' => true,
            'message' => "Checked out {$checkedOut} visitor(s)",
            'data' => [
                'checked_out' => $checkedOut,
                'already_checked_out' => $alreadyCheckedOut,
                'errors' => $errors,
            ],
        ]);
    }

    /**
     * Bulk delete visitors.
     */
    public function bulkDelete(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'visitor_ids' => 'required|array',
            'visitor_ids.*' => 'required|exists:visitors,id',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors(),
            ], 422);
        }

        $visitorIds = $request->visitor_ids;
        $deleted = 0;
        $errors = [];

        foreach ($visitorIds as $visitorId) {
            try {
                $visitor = Visitor::find($visitorId);
                if ($visitor) {
                    $visitor->delete();
                    $deleted++;
                }
            } catch (\Exception $e) {
                $errors[] = "Failed to delete visitor ID {$visitorId}: " . $e->getMessage();
            }
        }

        return response()->json([
            'success' => true,
            'message' => "Deleted {$deleted} visitor(s)",
            'data' => [
                'deleted' => $deleted,
                'errors' => $errors,
            ],
        ]);
    }

    private function formatVisitor(Visitor $visitor): array
    {
        return [
            'id' => (string) $visitor->id,
            'fullName' => $visitor->full_name,
            'phoneNumber' => $visitor->phone_number,
            'email' => $visitor->email,
            'companyName' => $visitor->company_name,
            'purpose' => $visitor->purpose,
            'hostId' => (string) $visitor->host_id,
            'hostName' => $visitor->host->name,
            'timeIn' => $visitor->time_in->toISOString(),
            'timeOut' => $visitor->time_out ? $visitor->time_out->toISOString() : null,
            'status' => $visitor->status,
        ];
    }
}
