<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Staff;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class StaffController extends Controller
{
    /**
     * Display a listing of the staff.
     */
    public function index(): JsonResponse
    {
        $staff = Staff::all();
        
        return response()->json([
            'success' => true,
            'data' => $staff->map(function ($member) {
                return [
                    'id' => (string) $member->id,
                    'name' => $member->name,
                    'department' => $member->department,
                ];
            }),
        ]);
    }

    /**
     * Store a newly created staff member.
     */
    public function store(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'department' => 'required|string|max:255',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors(),
            ], 422);
        }

        $staff = Staff::create([
            'name' => $request->name,
            'department' => $request->department,
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Staff member created successfully',
            'data' => [
                'id' => (string) $staff->id,
                'name' => $staff->name,
                'department' => $staff->department,
            ],
        ], 201);
    }

    /**
     * Update the specified staff member.
     */
    public function update(Request $request, string $id): JsonResponse
    {
        $staff = Staff::findOrFail($id);

        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'department' => 'required|string|max:255',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors(),
            ], 422);
        }

        $staff->update([
            'name' => $request->name,
            'department' => $request->department,
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Staff member updated successfully',
            'data' => [
                'id' => (string) $staff->id,
                'name' => $staff->name,
                'department' => $staff->department,
            ],
        ]);
    }

    /**
     * Remove the specified staff member.
     */
    public function destroy(string $id): JsonResponse
    {
        $staff = Staff::findOrFail($id);
        
        // Check if staff member has visitors (optimized with exists() instead of count())
        if ($staff->visitors()->exists()) {
            return response()->json([
                'success' => false,
                'message' => 'Cannot delete staff member with associated visitors. Please check out all visitors first.',
            ], 400);
        }

        $staff->delete();

        return response()->json([
            'success' => true,
            'message' => 'Staff member deleted successfully',
        ]);
    }
}
