<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\Cache;

class SettingsController extends Controller
{
    /**
     * Get all settings.
     * Uses caching to avoid reading files on every request.
     */
    public function index(): JsonResponse
    {
        try {
            // Cache settings for 5 minutes to improve performance
            $settings = Cache::remember('app_settings', 300, function () {
                // Try to load from storage first (preferred method)
                $mailConfig = [
                    'mailer' => 'smtp',
                    'host' => '',
                    'port' => '587',
                    'username' => '',
                    'password' => '', // Don't return password
                    'encryption' => 'tls',
                    'fromAddress' => '',
                    'fromName' => '',
                ];

                // Load from storage file if it exists
                try {
                    $storagePath = storage_path('app/mail_config.json');
                    if (file_exists($storagePath) && is_readable($storagePath)) {
                        $configContent = @file_get_contents($storagePath);
                        if ($configContent !== false) {
                            $savedConfig = json_decode($configContent, true);
                            if (is_array($savedConfig) && !empty($savedConfig)) {
                                $mailConfig['mailer'] = $savedConfig['mailer'] ?? $mailConfig['mailer'];
                                $mailConfig['host'] = $savedConfig['host'] ?? $mailConfig['host'];
                                $mailConfig['port'] = $savedConfig['port'] ?? $mailConfig['port'];
                                $mailConfig['username'] = $savedConfig['username'] ?? $mailConfig['username'];
                                $mailConfig['encryption'] = $savedConfig['encryption'] ?? $mailConfig['encryption'];
                                $mailConfig['fromAddress'] = $savedConfig['fromAddress'] ?? $mailConfig['fromAddress'];
                                $mailConfig['fromName'] = $savedConfig['fromName'] ?? $mailConfig['fromName'];
                                // Never return password
                                $mailConfig['password'] = '';
                            }
                        }
                    }
                } catch (\Exception $e) {
                    \Log::warning('Failed to load mail config from storage: ' . $e->getMessage());
                }

                // Fallback to .env file if storage file doesn't exist or host is empty
                if (empty($mailConfig['host'])) {
                    try {
                        $envFile = base_path('.env');
                        if (file_exists($envFile) && is_readable($envFile)) {
                            $envContent = @file_get_contents($envFile);
                            if ($envContent !== false) {
                                $lines = explode("\n", $envContent);
                                
                                foreach ($lines as $line) {
                                    $line = trim($line);
                                    if (empty($line) || strpos($line, '#') === 0 || strpos($line, '=') === false) {
                                        continue;
                                    }
                                    
                                    $parts = explode('=', $line, 2);
                                    if (count($parts) !== 2) {
                                        continue;
                                    }
                                    
                                    $key = trim($parts[0]);
                                    $value = trim($parts[1]);
                                    
                                    // Remove quotes if present
                                    if ((strlen($value) >= 2 && $value[0] === '"' && substr($value, -1) === '"') ||
                                        (strlen($value) >= 2 && $value[0] === "'" && substr($value, -1) === "'")) {
                                        $value = substr($value, 1, -1);
                                    }
                                    
                                    switch ($key) {
                                        case 'MAIL_MAILER':
                                            $mailConfig['mailer'] = $value;
                                            break;
                                        case 'MAIL_HOST':
                                            $mailConfig['host'] = $value;
                                            break;
                                        case 'MAIL_PORT':
                                            $mailConfig['port'] = $value;
                                            break;
                                        case 'MAIL_USERNAME':
                                            $mailConfig['username'] = $value;
                                            break;
                                        case 'MAIL_ENCRYPTION':
                                            $mailConfig['encryption'] = $value;
                                            break;
                                        case 'MAIL_FROM_ADDRESS':
                                            $mailConfig['fromAddress'] = $value;
                                            break;
                                        case 'MAIL_FROM_NAME':
                                            $mailConfig['fromName'] = $value;
                                            break;
                                    }
                                }
                            }
                        }
                    } catch (\Exception $e) {
                        \Log::warning('Failed to read .env file: ' . $e->getMessage());
                    }
                }

                // Get checkout message
                $checkoutMessage = 'Dear {visitorName}, thank you for visiting {companyName} today! We hope you had a pleasant experience. Safe travels!';
                try {
                    $messagePath = storage_path('app/checkout_message.txt');
                    if (file_exists($messagePath) && is_readable($messagePath)) {
                        $messageContent = @file_get_contents($messagePath);
                        if ($messageContent !== false) {
                            $checkoutMessage = $messageContent;
                        }
                    }
                } catch (\Exception $e) {
                    \Log::warning('Failed to load checkout message: ' . $e->getMessage());
                }

                return [
                    'mailConfig' => $mailConfig,
                    'checkoutMessage' => $checkoutMessage,
                ];
            });

            return response()->json([
                'success' => true,
                'data' => $settings,
            ]);
        } catch (\Exception $e) {
            \Log::error('Error in SettingsController::index: ' . $e->getMessage());
            \Log::error('Stack trace: ' . $e->getTraceAsString());
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to load settings: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Change user password.
     */
    public function changePassword(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'currentPassword' => 'required|string',
            'newPassword' => 'required|string|min:8|regex:/^(?=.*[a-z])(?=.*[A-Z])(?=.*\d)/',
        ], [
            'newPassword.regex' => 'Password must contain at least one uppercase letter, one lowercase letter, and one number.',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors(),
            ], 422);
        }

        $user = Auth::user();

        if (!Hash::check($request->currentPassword, $user->password)) {
            return response()->json([
                'success' => false,
                'message' => 'Current password is incorrect',
            ], 400);
        }

        $user->password = Hash::make($request->newPassword);
        $user->save();

        return response()->json([
            'success' => true,
            'message' => 'Password changed successfully',
        ]);
    }

    /**
     * Save mail configuration.
     */
    public function saveMailConfig(Request $request): JsonResponse
    {
        try {
            $validator = Validator::make($request->all(), [
                'mailer' => 'required|string',
                'host' => 'required|string',
                'port' => 'required|string',
                'username' => 'required|string',
                'password' => 'required|string',
                'encryption' => 'required|string|in:tls,ssl',
                'fromAddress' => 'required|email',
                'fromName' => 'required|string',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'errors' => $validator->errors(),
                ], 422);
            }

            // Store mail config in storage instead of .env for better reliability
            $mailConfigData = [
                'mailer' => $request->mailer,
                'host' => $request->host,
                'port' => $request->port,
                'username' => $request->username,
                'password' => $request->password,
                'encryption' => $request->encryption,
                'fromAddress' => $request->fromAddress,
                'fromName' => $request->fromName,
            ];

            // Save to storage file
            Storage::disk('local')->put('mail_config.json', json_encode($mailConfigData, JSON_PRETTY_PRINT));
            
            // Clear cache so new settings are loaded on next request
            Cache::forget('app_settings');

            // Also try to update .env file (optional, for compatibility)
            try {
                $envFile = base_path('.env');
                if (file_exists($envFile) && is_writable($envFile)) {
                    $envContent = file_get_contents($envFile);
                    
                    $updates = [
                        'MAIL_MAILER' => $request->mailer,
                        'MAIL_HOST' => $request->host,
                        'MAIL_PORT' => $request->port,
                        'MAIL_USERNAME' => $request->username,
                        'MAIL_PASSWORD' => $request->password,
                        'MAIL_ENCRYPTION' => $request->encryption,
                        'MAIL_FROM_ADDRESS' => $request->fromAddress,
                        'MAIL_FROM_NAME' => $request->fromName,
                    ];

                    foreach ($updates as $key => $value) {
                        $escapedValue = preg_match('/\s|#|"|\'/', $value) ? '"' . addcslashes($value, '"') . '"' : $value;
                        $pattern = "/^(\s*#?\s*){$key}\s*=.*/m";
                        $replacement = "{$key}={$escapedValue}";
                        
                        if (preg_match($pattern, $envContent)) {
                            $envContent = preg_replace($pattern, $replacement, $envContent);
                        } else {
                            $envContent .= "\n{$replacement}";
                        }
                    }

                    file_put_contents($envFile, $envContent);
                }
            } catch (\Exception $e) {
                \Log::warning('Failed to update .env file: ' . $e->getMessage());
            }

            return response()->json([
                'success' => true,
                'message' => 'Mail configuration saved successfully',
            ]);
        } catch (\Exception $e) {
            \Log::error('Error saving mail config: ' . $e->getMessage());
            \Log::error('Stack trace: ' . $e->getTraceAsString());
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to save mail configuration: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Save checkout message.
     */
    public function saveCheckoutMessage(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'message' => 'required|string|max:1000',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors(),
            ], 422);
        }

        Storage::disk('local')->put('checkout_message.txt', $request->message);
        
        // Clear cache so new message is loaded on next request
        Cache::forget('app_settings');

        return response()->json([
            'success' => true,
            'message' => 'Checkout message saved successfully',
        ]);
    }
}
