<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\ScheduledVisit;
use App\Models\Visitor;
use App\Services\AuditLogService;
use App\Services\NotificationService;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;

class ScheduledVisitController extends Controller
{
    /**
     * Get all scheduled visits (with filters)
     */
    public function index(Request $request): JsonResponse
    {
        $query = ScheduledVisit::with('host');

        // Filter by status
        if ($request->has('status')) {
            $query->where('status', $request->status);
        }

        // Filter by date range
        if ($request->has('date_from')) {
            $query->whereDate('scheduled_date', '>=', $request->date_from);
        }
        if ($request->has('date_to')) {
            $query->whereDate('scheduled_date', '<=', $request->date_to);
        }

        // Filter by host
        if ($request->has('host_id')) {
            $query->where('host_id', $request->host_id);
        }

        // Filter by upcoming/past
        if ($request->has('upcoming') && $request->upcoming) {
            $query->where('status', 'scheduled')
                  ->whereDate('scheduled_date', '>=', now()->toDateString());
        }

        // Search
        if ($request->has('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('visitor_name', 'like', "%{$search}%")
                  ->orWhere('visitor_email', 'like', "%{$search}%")
                  ->orWhere('visitor_phone', 'like', "%{$search}%")
                  ->orWhere('company_name', 'like', "%{$search}%");
            });
        }

        // Sorting
        $sortBy = $request->get('sort_by', 'scheduled_date');
        $sortOrder = $request->get('sort_order', 'asc');
        $query->orderBy($sortBy, $sortOrder);

        $perPage = min((int) $request->get('per_page', 50), 100);
        $page = (int) $request->get('page', 1);
        $paginated = $query->paginate($perPage, ['*'], 'page', $page);

        return response()->json([
            'success' => true,
            'data' => $paginated->map(function ($visit) {
                return $this->formatScheduledVisit($visit);
            }),
            'meta' => [
                'total' => $paginated->total(),
                'per_page' => $paginated->perPage(),
                'current_page' => $paginated->currentPage(),
                'last_page' => $paginated->lastPage(),
            ],
        ]);
    }

    /**
     * Create a new scheduled visit (public endpoint for visitors)
     */
    public function store(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'visitor_name' => 'required|string|max:255',
            'visitor_email' => 'nullable|email|max:255',
            'visitor_phone' => 'required|string|max:20',
            'company_name' => 'required|string|max:255',
            'purpose' => 'required|string|max:255',
            'host_id' => 'required|exists:staff,id',
            'scheduled_date' => 'required|date|after_or_equal:today',
            'scheduled_time' => 'required|date_format:H:i',
            'notes' => 'nullable|string|max:1000',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors(),
            ], 422);
        }

        // Check if host is available at that time (optional - can be enhanced)
        $conflict = ScheduledVisit::where('host_id', $request->host_id)
            ->where('scheduled_date', $request->scheduled_date)
            ->where('scheduled_time', $request->scheduled_time)
            ->where('status', 'scheduled')
            ->exists();

        if ($conflict) {
            return response()->json([
                'success' => false,
                'message' => 'Host is already booked at this time. Please choose another time.',
            ], 409);
        }

        $scheduledVisit = ScheduledVisit::create([
            'visitor_name' => $request->visitor_name,
            'visitor_email' => $request->visitor_email,
            'visitor_phone' => $request->visitor_phone,
            'company_name' => $request->company_name,
            'purpose' => $request->purpose,
            'host_id' => $request->host_id,
            'scheduled_date' => $request->scheduled_date,
            'scheduled_time' => $request->scheduled_time,
            'status' => 'scheduled',
            'notes' => $request->notes,
            'confirmation_token' => Str::random(64),
        ]);

        $scheduledVisit->load('host');

        // Log action
        AuditLogService::log('created', $scheduledVisit, null, $scheduledVisit->toArray(), 'Scheduled visit created', $request);

        // Send confirmation email to visitor
        if ($scheduledVisit->visitor_email) {
            try {
                NotificationService::notifyVisitorScheduled($scheduledVisit);
            } catch (\Exception $e) {
                \Log::warning('Failed to send scheduled visit confirmation: ' . $e->getMessage());
            }
        }

        // Notify host
        try {
            NotificationService::notifyHostScheduled($scheduledVisit);
        } catch (\Exception $e) {
            \Log::warning('Failed to send host notification: ' . $e->getMessage());
        }

        return response()->json([
            'success' => true,
            'message' => 'Visit scheduled successfully. Confirmation email sent.',
            'data' => $this->formatScheduledVisit($scheduledVisit),
        ], 201);
    }

    /**
     * Get a specific scheduled visit
     */
    public function show(string $id): JsonResponse
    {
        $visit = ScheduledVisit::with('host', 'checkIn')->findOrFail($id);

        return response()->json([
            'success' => true,
            'data' => $this->formatScheduledVisit($visit),
        ]);
    }

    /**
     * Update scheduled visit (admin only)
     */
    public function update(Request $request, string $id): JsonResponse
    {
        $visit = ScheduledVisit::findOrFail($id);

        $validator = Validator::make($request->all(), [
            'visitor_name' => 'sometimes|string|max:255',
            'visitor_email' => 'nullable|email|max:255',
            'visitor_phone' => 'sometimes|string|max:20',
            'company_name' => 'sometimes|string|max:255',
            'purpose' => 'sometimes|string|max:255',
            'host_id' => 'sometimes|exists:staff,id',
            'scheduled_date' => 'sometimes|date',
            'scheduled_time' => 'sometimes|date_format:H:i',
            'status' => 'sometimes|in:scheduled,checked_in,cancelled,no_show',
            'notes' => 'nullable|string|max:1000',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors(),
            ], 422);
        }

        $oldValues = $visit->toArray();
        $visit->update($request->only([
            'visitor_name',
            'visitor_email',
            'visitor_phone',
            'company_name',
            'purpose',
            'host_id',
            'scheduled_date',
            'scheduled_time',
            'status',
            'notes',
        ]));

        // Log action
        AuditLogService::log('updated', $visit, $oldValues, $visit->toArray(), 'Scheduled visit updated', $request);

        return response()->json([
            'success' => true,
            'message' => 'Scheduled visit updated successfully',
            'data' => $this->formatScheduledVisit($visit->fresh()),
        ]);
    }

    /**
     * Cancel a scheduled visit
     */
    public function cancel(string $id): JsonResponse
    {
        $visit = ScheduledVisit::findOrFail($id);

        if ($visit->status === 'checked_in') {
            return response()->json([
                'success' => false,
                'message' => 'Cannot cancel a visit that has already been checked in',
            ], 400);
        }

        $visit->update(['status' => 'cancelled']);

        AuditLogService::log('updated', $visit, ['status' => $visit->getOriginal('status')], ['status' => 'cancelled'], 'Scheduled visit cancelled', request());

        return response()->json([
            'success' => true,
            'message' => 'Scheduled visit cancelled successfully',
            'data' => $this->formatScheduledVisit($visit->fresh()),
        ]);
    }

    /**
     * Check in a scheduled visit (converts to actual visitor)
     */
    public function checkIn(Request $request, string $id): JsonResponse
    {
        $scheduledVisit = ScheduledVisit::with('host')->findOrFail($id);

        if ($scheduledVisit->status === 'checked_in') {
            return response()->json([
                'success' => false,
                'message' => 'This visit has already been checked in',
            ], 400);
        }

        // Create actual visitor record
        $visitor = Visitor::create([
            'full_name' => $scheduledVisit->visitor_name,
            'phone_number' => $scheduledVisit->visitor_phone,
            'email' => $scheduledVisit->visitor_email,
            'company_name' => $scheduledVisit->company_name,
            'purpose' => $scheduledVisit->purpose,
            'host_id' => $scheduledVisit->host_id,
            'time_in' => now(),
            'status' => 'checked-in',
        ]);

        // Update scheduled visit
        $scheduledVisit->update([
            'status' => 'checked_in',
            'check_in_id' => $visitor->id,
        ]);

        $visitor->load('host');

        // Log actions
        AuditLogService::log('created', $visitor, null, $visitor->toArray(), 'Visitor checked in from scheduled visit', $request);
        AuditLogService::log('updated', $scheduledVisit, ['status' => 'scheduled'], ['status' => 'checked_in'], 'Scheduled visit checked in', $request);

        // Send notifications
        try {
            NotificationService::notifyHostOnArrival($visitor);
            NotificationService::notifyAdminOnCheckIn($visitor);
        } catch (\Exception $e) {
            \Log::warning('Failed to send notifications: ' . $e->getMessage());
        }

        // Format visitor response
        $formattedVisitor = [
            'id' => (string) $visitor->id,
            'fullName' => $visitor->full_name,
            'phoneNumber' => $visitor->phone_number,
            'email' => $visitor->email,
            'companyName' => $visitor->company_name,
            'purpose' => $visitor->purpose,
            'hostId' => (string) $visitor->host_id,
            'hostName' => $visitor->host->name,
            'timeIn' => $visitor->time_in->toISOString(),
            'timeOut' => null,
            'status' => $visitor->status,
        ];
        
        return response()->json([
            'success' => true,
            'message' => 'Scheduled visit checked in successfully',
            'data' => [
                'scheduled_visit' => $this->formatScheduledVisit($scheduledVisit->fresh()),
                'visitor' => $formattedVisitor,
            ],
        ]);
    }

    /**
     * Format scheduled visit for API response
     */
    private function formatScheduledVisit(ScheduledVisit $visit): array
    {
        return [
            'id' => (string) $visit->id,
            'visitorName' => $visit->visitor_name,
            'visitorEmail' => $visit->visitor_email,
            'visitorPhone' => $visit->visitor_phone,
            'companyName' => $visit->company_name,
            'purpose' => $visit->purpose,
            'hostId' => (string) $visit->host_id,
            'hostName' => $visit->host->name,
            'hostDepartment' => $visit->host->department,
            'scheduledDate' => $visit->scheduled_date->format('Y-m-d'),
            'scheduledTime' => $visit->scheduled_time,
            'scheduledDateTime' => $visit->scheduled_date->format('Y-m-d') . ' ' . $visit->scheduled_time,
            'status' => $visit->status,
            'checkInId' => $visit->check_in_id ? (string) $visit->check_in_id : null,
            'notes' => $visit->notes,
            'isPast' => $visit->isPast(),
            'isToday' => $visit->isToday(),
            'isUpcoming' => $visit->isUpcoming(),
            'createdAt' => $visit->created_at->toISOString(),
            'updatedAt' => $visit->updated_at->toISOString(),
        ];
    }
}
