<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Artisan;
use ZipArchive;

class BackupController extends Controller
{
    /**
     * Create a database backup
     */
    public function create(Request $request): JsonResponse
    {
        try {
            $backupType = $request->get('type', 'database'); // database, full
            
            if ($backupType === 'full') {
                return $this->createFullBackup();
            } else {
                return $this->createDatabaseBackup();
            }
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to create backup: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Create database backup only
     */
    private function createDatabaseBackup(): JsonResponse
    {
        try {
            $database = config('database.connections.mysql.database');
            $username = config('database.connections.mysql.username');
            $password = config('database.connections.mysql.password');
            $host = config('database.connections.mysql.host');
            $port = config('database.connections.mysql.port', 3306);
            
            $filename = 'backup_' . date('Y-m-d_His') . '.sql';
            $filepath = storage_path('app/backups/' . $filename);
            
            // Create backups directory if it doesn't exist
            $backupsDir = storage_path('app/backups');
            if (!file_exists($backupsDir)) {
                mkdir($backupsDir, 0755, true);
            }

            // Check if mysqldump is available
            $mysqldumpPath = $this->findMysqldump();
            if (!$mysqldumpPath) {
                return response()->json([
                    'success' => false,
                    'message' => 'mysqldump command not found. Please ensure MySQL client tools are installed.',
                ], 500);
            }

            // Create mysqldump command with proper password handling
            // Use --password= or environment variable to avoid password in process list
            $command = sprintf(
                '%s -h %s -P %s -u %s --password=%s %s > %s 2>&1',
                escapeshellarg($mysqldumpPath),
                escapeshellarg($host),
                escapeshellarg($port),
                escapeshellarg($username),
                escapeshellarg($password),
                escapeshellarg($database),
                escapeshellarg($filepath)
            );

            // Execute command and capture output
            exec($command, $output, $returnVar);

            // Check if file was created and has content
            if ($returnVar !== 0 || !file_exists($filepath) || filesize($filepath) === 0) {
                $errorMsg = !empty($output) ? implode("\n", $output) : 'Unknown error';
                \Log::error('Backup failed', [
                    'command' => $command,
                    'return_code' => $returnVar,
                    'output' => $output,
                ]);
                
                // Clean up empty file if it exists
                if (file_exists($filepath) && filesize($filepath) === 0) {
                    unlink($filepath);
                }
                
                return response()->json([
                    'success' => false,
                    'message' => 'Failed to create database backup: ' . $errorMsg,
                ], 500);
            }

            // Get file size
            $fileSize = filesize($filepath);

            return response()->json([
                'success' => true,
                'message' => 'Backup created successfully',
                'data' => [
                    'filename' => $filename,
                    'filepath' => $filepath,
                    'size' => $fileSize,
                    'sizeFormatted' => $this->formatBytes($fileSize),
                    'createdAt' => now()->toISOString(),
                    'type' => 'database',
                ],
            ]);
        } catch (\Exception $e) {
            \Log::error('Backup exception', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to create backup: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Find mysqldump executable path
     */
    private function findMysqldump(): ?string
    {
        // Common paths for mysqldump
        $paths = [
            '/usr/bin/mysqldump',
            '/usr/local/bin/mysqldump',
            '/opt/homebrew/bin/mysqldump', // macOS Homebrew
            '/Applications/XAMPP/xamppfiles/bin/mysqldump', // XAMPP
            'mysqldump', // In PATH
        ];

        foreach ($paths as $path) {
            if ($path === 'mysqldump') {
                // Check if it's in PATH
                exec('which mysqldump 2>/dev/null', $output, $returnVar);
                if ($returnVar === 0 && !empty($output[0])) {
                    return $output[0];
                }
            } elseif (file_exists($path) && is_executable($path)) {
                return $path;
            }
        }

        return null;
    }

    /**
     * Create full backup (database + files)
     */
    private function createFullBackup(): JsonResponse
    {
        // Create database backup first
        $dbBackup = $this->createDatabaseBackup();
        if (!$dbBackup->getData(true)['success']) {
            return $dbBackup;
        }

        $dbFilename = $dbBackup->getData(true)['data']['filename'];
        $dbFilepath = $dbBackup->getData(true)['data']['filepath'];

        // Create ZIP archive
        $zipFilename = 'full_backup_' . date('Y-m-d_His') . '.zip';
        $zipFilepath = storage_path('app/backups/' . $zipFilename);
        
        $zip = new ZipArchive();
        if ($zip->open($zipFilepath, ZipArchive::CREATE) !== TRUE) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to create backup archive',
            ], 500);
        }

        // Add database backup
        $zip->addFile($dbFilepath, 'database/' . $dbFilename);

        // Add important files (config, storage files)
        $filesToBackup = [
            'storage/app/mail_config.json',
            'storage/app/checkout_message.txt',
        ];

        foreach ($filesToBackup as $file) {
            $fullPath = base_path($file);
            if (file_exists($fullPath)) {
                $zip->addFile($fullPath, $file);
            }
        }

        $zip->close();

        $fileSize = filesize($zipFilepath);

        return response()->json([
            'success' => true,
            'message' => 'Full backup created successfully',
            'data' => [
                'filename' => $zipFilename,
                'filepath' => $zipFilepath,
                'size' => $fileSize,
                'sizeFormatted' => $this->formatBytes($fileSize),
                'createdAt' => now()->toISOString(),
                'type' => 'full',
            ],
        ]);
    }

    /**
     * List all backups
     */
    public function list(): JsonResponse
    {
        $backupsDir = storage_path('app/backups');
        
        if (!file_exists($backupsDir)) {
            return response()->json([
                'success' => true,
                'data' => [],
            ]);
        }

        $files = glob($backupsDir . '/*.{sql,zip}', GLOB_BRACE);
        $backups = [];

        foreach ($files as $file) {
            $backups[] = [
                'filename' => basename($file),
                'filepath' => $file,
                'size' => filesize($file),
                'sizeFormatted' => $this->formatBytes(filesize($file)),
                'createdAt' => date('Y-m-d H:i:s', filemtime($file)),
                'type' => pathinfo($file, PATHINFO_EXTENSION) === 'zip' ? 'full' : 'database',
            ];
        }

        // Sort by creation date (newest first)
        usort($backups, function ($a, $b) {
            return strtotime($b['createdAt']) - strtotime($a['createdAt']);
        });

        return response()->json([
            'success' => true,
            'data' => $backups,
        ]);
    }

    /**
     * Download a backup file
     */
    public function download(string $filename): \Illuminate\Http\Response
    {
        $filepath = storage_path('app/backups/' . $filename);

        if (!file_exists($filepath)) {
            abort(404, 'Backup file not found');
        }

        return response()->download($filepath, $filename);
    }

    /**
     * Delete a backup
     */
    public function delete(string $filename): JsonResponse
    {
        $filepath = storage_path('app/backups/' . $filename);

        if (!file_exists($filepath)) {
            return response()->json([
                'success' => false,
                'message' => 'Backup file not found',
            ], 404);
        }

        if (unlink($filepath)) {
            return response()->json([
                'success' => true,
                'message' => 'Backup deleted successfully',
            ]);
        }

        return response()->json([
            'success' => false,
            'message' => 'Failed to delete backup',
        ], 500);
    }

    /**
     * Format bytes to human readable format
     */
    private function formatBytes(int $bytes, int $precision = 2): string
    {
        $units = ['B', 'KB', 'MB', 'GB', 'TB'];
        
        for ($i = 0; $bytes > 1024 && $i < count($units) - 1; $i++) {
            $bytes /= 1024;
        }
        
        return round($bytes, $precision) . ' ' . $units[$i];
    }
}
