<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Visitor;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class AnalyticsController extends Controller
{
    /**
     * Get comprehensive analytics dashboard data
     */
    public function dashboard(Request $request): JsonResponse
    {
        $dateFrom = $request->get('date_from', now()->subDays(30)->format('Y-m-d'));
        $dateTo = $request->get('date_to', now()->format('Y-m-d'));

        return response()->json([
            'success' => true,
            'data' => [
                'overview' => $this->getOverview($dateFrom, $dateTo),
                'dailyTrends' => $this->getDailyTrends($dateFrom, $dateTo),
                'hourlyPatterns' => $this->getHourlyPatterns($dateFrom, $dateTo),
                'hostStats' => $this->getHostStats($dateFrom, $dateTo),
                'purposeStats' => $this->getPurposeStats($dateFrom, $dateTo),
                'peakTimes' => $this->getPeakTimes($dateFrom, $dateTo),
                'comparison' => $this->getComparison($dateFrom, $dateTo),
            ],
        ]);
    }

    /**
     * Get overview statistics
     */
    private function getOverview(string $dateFrom, string $dateTo): array
    {
        $total = Visitor::whereBetween('time_in', [$dateFrom, $dateTo . ' 23:59:59'])->count();
        $checkedIn = Visitor::whereBetween('time_in', [$dateFrom, $dateTo . ' 23:59:59'])
            ->where('status', 'checked-in')
            ->count();
        $checkedOut = Visitor::whereBetween('time_in', [$dateFrom, $dateTo . ' 23:59:59'])
            ->where('status', 'checked-out')
            ->count();
        
        $avgDuration = Visitor::whereBetween('time_in', [$dateFrom, $dateTo . ' 23:59:59'])
            ->whereNotNull('time_out')
            ->selectRaw('AVG(TIMESTAMPDIFF(MINUTE, time_in, time_out)) as avg_duration')
            ->value('avg_duration') ?? 0;

        $uniqueVisitors = Visitor::whereBetween('time_in', [$dateFrom, $dateTo . ' 23:59:59'])
            ->distinct('email')
            ->whereNotNull('email')
            ->count('email');

        return [
            'total' => $total,
            'checkedIn' => $checkedIn,
            'checkedOut' => $checkedOut,
            'avgDurationMinutes' => round($avgDuration, 2),
            'uniqueVisitors' => $uniqueVisitors,
        ];
    }

    /**
     * Get daily trends
     */
    private function getDailyTrends(string $dateFrom, string $dateTo): array
    {
        $trends = Visitor::whereBetween('time_in', [$dateFrom, $dateTo . ' 23:59:59'])
            ->selectRaw('DATE(time_in) as date, COUNT(*) as count')
            ->groupBy('date')
            ->orderBy('date')
            ->get()
            ->map(function ($item) {
                return [
                    'date' => $item->date,
                    'count' => $item->count,
                ];
            });

        return $trends->toArray();
    }

    /**
     * Get hourly patterns (average visitors per hour)
     */
    private function getHourlyPatterns(string $dateFrom, string $dateTo): array
    {
        $patterns = Visitor::whereBetween('time_in', [$dateFrom, $dateTo . ' 23:59:59'])
            ->selectRaw('HOUR(time_in) as hour, COUNT(*) as count')
            ->groupBy('hour')
            ->orderBy('hour')
            ->get();

        // Initialize all hours
        $hourlyData = [];
        for ($hour = 0; $hour < 24; $hour++) {
            $hourlyData[$hour] = 0;
        }

        // Fill in actual data
        foreach ($patterns as $pattern) {
            $hourlyData[$pattern->hour] = $pattern->count;
        }

        // Convert to array format
        $result = [];
        foreach ($hourlyData as $hour => $count) {
            $hourLabel = $hour < 12 
                ? ($hour === 0 ? '12AM' : $hour . 'AM')
                : ($hour === 12 ? '12PM' : ($hour - 12) . 'PM');
            
            $result[] = [
                'hour' => $hour,
                'label' => $hourLabel,
                'count' => $count,
            ];
        }

        return $result;
    }

    /**
     * Get host statistics
     */
    private function getHostStats(string $dateFrom, string $dateTo): array
    {
        $stats = Visitor::whereBetween('time_in', [$dateFrom, $dateTo . ' 23:59:59'])
            ->join('staff', 'visitors.host_id', '=', 'staff.id')
            ->selectRaw('staff.id, staff.name, staff.department, COUNT(visitors.id) as visitor_count')
            ->groupBy('staff.id', 'staff.name', 'staff.department')
            ->orderByDesc('visitor_count')
            ->limit(10)
            ->get()
            ->map(function ($item) {
                return [
                    'hostId' => (string) $item->id,
                    'hostName' => $item->name,
                    'department' => $item->department,
                    'visitorCount' => $item->visitor_count,
                ];
            });

        return $stats->toArray();
    }

    /**
     * Get purpose statistics
     */
    private function getPurposeStats(string $dateFrom, string $dateTo): array
    {
        $stats = Visitor::whereBetween('time_in', [$dateFrom, $dateTo . ' 23:59:59'])
            ->selectRaw('purpose, COUNT(*) as count')
            ->groupBy('purpose')
            ->orderByDesc('count')
            ->get()
            ->map(function ($item) {
                return [
                    'purpose' => $item->purpose,
                    'count' => $item->count,
                ];
            });

        return $stats->toArray();
    }

    /**
     * Get peak times analysis
     */
    private function getPeakTimes(string $dateFrom, string $dateTo): array
    {
        // Peak day of week
        $peakDay = Visitor::whereBetween('time_in', [$dateFrom, $dateTo . ' 23:59:59'])
            ->selectRaw('DAYNAME(time_in) as day_name, DAYOFWEEK(time_in) as day_num, COUNT(*) as count')
            ->groupBy('day_name', 'day_num')
            ->orderByDesc('count')
            ->first();

        // Peak hour
        $peakHour = Visitor::whereBetween('time_in', [$dateFrom, $dateTo . ' 23:59:59'])
            ->selectRaw('HOUR(time_in) as hour, COUNT(*) as count')
            ->groupBy('hour')
            ->orderByDesc('count')
            ->first();

        // Busiest day
        $busiestDay = Visitor::whereBetween('time_in', [$dateFrom, $dateTo . ' 23:59:59'])
            ->selectRaw('DATE(time_in) as date, COUNT(*) as count')
            ->groupBy('date')
            ->orderByDesc('count')
            ->first();

        return [
            'peakDayOfWeek' => $peakDay ? [
                'day' => $peakDay->day_name,
                'count' => $peakDay->count,
            ] : null,
            'peakHour' => $peakHour ? [
                'hour' => $peakHour->hour,
                'label' => $peakHour->hour < 12 
                    ? ($peakHour->hour === 0 ? '12AM' : $peakHour->hour . 'AM')
                    : ($peakHour->hour === 12 ? '12PM' : ($peakHour->hour - 12) . 'PM'),
                'count' => $peakHour->count,
            ] : null,
            'busiestDay' => $busiestDay ? [
                'date' => $busiestDay->date,
                'count' => $busiestDay->count,
            ] : null,
        ];
    }

    /**
     * Get comparison data (this period vs previous period)
     */
    private function getComparison(string $dateFrom, string $dateTo): array
    {
        $currentStart = $dateFrom;
        $currentEnd = $dateTo;
        
        // Calculate previous period (same duration)
        $startDate = \Carbon\Carbon::parse($dateFrom);
        $endDate = \Carbon\Carbon::parse($dateTo);
        $daysDiff = $startDate->diffInDays($endDate);
        
        $previousStart = $startDate->copy()->subDays($daysDiff + 1)->format('Y-m-d');
        $previousEnd = $startDate->copy()->subDay()->format('Y-m-d');

        // Current period stats
        $currentTotal = Visitor::whereBetween('time_in', [$currentStart, $currentEnd . ' 23:59:59'])->count();
        $currentCheckedIn = Visitor::whereBetween('time_in', [$currentStart, $currentEnd . ' 23:59:59'])
            ->where('status', 'checked-in')
            ->count();

        // Previous period stats
        $previousTotal = Visitor::whereBetween('time_in', [$previousStart, $previousEnd . ' 23:59:59'])->count();
        $previousCheckedIn = Visitor::whereBetween('time_in', [$previousStart, $previousEnd . ' 23:59:59'])
            ->where('status', 'checked-in')
            ->count();

        // Calculate changes
        $totalChange = $previousTotal > 0 
            ? round((($currentTotal - $previousTotal) / $previousTotal) * 100, 2)
            : ($currentTotal > 0 ? 100 : 0);

        $checkedInChange = $previousCheckedIn > 0
            ? round((($currentCheckedIn - $previousCheckedIn) / $previousCheckedIn) * 100, 2)
            : ($currentCheckedIn > 0 ? 100 : 0);

        return [
            'current' => [
                'total' => $currentTotal,
                'checkedIn' => $currentCheckedIn,
                'period' => [
                    'from' => $currentStart,
                    'to' => $currentEnd,
                ],
            ],
            'previous' => [
                'total' => $previousTotal,
                'checkedIn' => $previousCheckedIn,
                'period' => [
                    'from' => $previousStart,
                    'to' => $previousEnd,
                ],
            ],
            'changes' => [
                'total' => [
                    'value' => $currentTotal - $previousTotal,
                    'percentage' => $totalChange,
                ],
                'checkedIn' => [
                    'value' => $currentCheckedIn - $previousCheckedIn,
                    'percentage' => $checkedInChange,
                ],
            ],
        ];
    }

    /**
     * Get weekly trends
     */
    public function weeklyTrends(Request $request): JsonResponse
    {
        $weeks = (int) $request->get('weeks', 4);
        $endDate = now();
        $startDate = now()->subWeeks($weeks);

        $trends = Visitor::whereBetween('time_in', [$startDate, $endDate])
            ->selectRaw('YEARWEEK(time_in) as week, COUNT(*) as count')
            ->groupBy('week')
            ->orderBy('week')
            ->get()
            ->map(function ($item) {
                return [
                    'week' => $item->week,
                    'count' => $item->count,
                ];
            });

        return response()->json([
            'success' => true,
            'data' => $trends,
        ]);
    }

    /**
     * Get monthly trends
     */
    public function monthlyTrends(Request $request): JsonResponse
    {
        $months = (int) $request->get('months', 12);
        $endDate = now();
        $startDate = now()->subMonths($months);

        $trends = Visitor::whereBetween('time_in', [$startDate, $endDate])
            ->selectRaw('DATE_FORMAT(time_in, "%Y-%m") as month, COUNT(*) as count')
            ->groupBy('month')
            ->orderBy('month')
            ->get()
            ->map(function ($item) {
                return [
                    'month' => $item->month,
                    'count' => $item->count,
                ];
            });

        return response()->json([
            'success' => true,
            'data' => $trends,
        ]);
    }
}
